# -*- perl -*-

# Copyright (c) 2002 by Jeff Weisberg
# Author: Jeff Weisberg <argus @ tcp4me.com>
# Date: 2002-Apr-02 09:47 (EST)
# Function: BaseIO class
#
# $Id: BaseIO.pm,v 1.29 2003/04/07 19:39:05 jaw Exp $

# low-lovel I/O non-blocking, select-loop

package BaseIO;
use Fcntl;
use POSIX qw(:errno_h);
# use the high resolution timer if we have it, and we aren't profiling
# currently, only Service elapsed time uses it
BEGIN { eval{ require Time::HiRes; import Time::HiRes qw(time) }
	unless defined &DB::DB; }

my @byfile = (); 	 # objects by fd
my @byfile_dbg = ();	 # for debugging...
   @bytime = (); 	 # { time obj func args text } sorted by time
			 # - global so that Ping::start can muck around...
my $rfds   = "\0\0\0\0"; # fds that want read
my $wfds   = "\0\0\0\0"; # fds that want write
my @timeouts = (); 	 # objects sorted by timeout

$doc = {
    package => __PACKAGE__,
    file    => __FILE__,
    isa     => [ ],
    methods => {
	wantread   => {
	    descr => 'set the want read parameter',
	},
	wantwrit   => {
	    descr => 'set the want write parameter',
	},
	settimeout => {
	    descr => 'set the timeout time',
	},
	readable   => {
	    descr => 'Called once the object becomes readable',
	    attrs => ['virtual'],
	},
	writable   => {
	    descr => 'Called once the object becomes writable',
	    attrs => ['virtual'],
	},
	timeout    => {
	    descr => 'Called once the object timeouts waiting to become readable or writable',
	    attrs => ['virtual'],
	},
	add_timed_func => {
	    descr => 'Add a function to run at a specified time',
	},
	clear_timed_funcs => {
	    descr => 'remove all currently scheduled timed functions',
	},
	shutdown   => {
	    descr => 'Called to shutdown the object',
	},
    },
    fields => {
	wantread   => {
	    descr => 'This object is waiting for data to read',
	    attrs => ['internal'],
	},
	wantwrit   => {
	    descr => 'This object is waiting to output data',
	    attrs => ['internal'],
	},
	fd         => {
	    descr => 'The objects filehandle',
	    attrs => ['internal'],
	},
	type       => {
	    descr => 'descriptive name of object type, for debugging',
	},
	timeout    => {
	    descr => 'When should object should stop waiting',
	    attrs => ['internal'],
	},
	opentime => {
	    descr => 'When the file descriptor was last opened',
	    #         not cleared on close, for debugging
	},
    },
};

################################################################
# public object methods
################################################################

sub wantread {
    my $me = shift;
    my $wm = shift;

    vec($rfds, fileno($me->{fd}), 1) = $wm;
    $me->{wantread} = $wm;
}

sub wantwrit {
    my $me = shift;
    my $wm = shift;

    vec($wfds, fileno($me->{fd}), 1) = $wm;
    $me->{wantwrit} = $wm;
}

sub settimeout {
    my $me = shift;
    my $t  = shift;
    my( %ts );

    # arg is delta, make absolute
    # 0 removes timeout
    $t += $^T if $t;
    # return if unchanged
    return if $me->{timeout} && ($t == $me->{timeout});

    $me->{timeout} = $t;
    
    if( $t ){
	# add, unique, sort
	push @timeouts, $me;
	foreach (@timeouts){
	    $ts{ fileno($_->{fd}) } = $_;
#	    print STDERR "$_->{type} ", ($_->can('unique') ? $_->unique() : ''),
#	    " [$_->{fd}], $_->{timeout}\n"
#		unless $_->{fd} && $_->{timeout};
	}
	@timeouts = sort { $a->{timeout} <=> $b->{timeout} } ( values %ts );
	# the astute reader is probably asking, "you go through all sorts of
	# hoops to speed up maintaining @bytime, shouldn't you be doing that
	# here as well?"
	# A: no, @bytime has lots and lots of entries, @timeouts typically
	# has only a few entries
    }else{
	# remove timeout
	@timeouts = grep { $_ != $me } @timeouts;
    }
}

# initialize fd
sub baseio_init {
    my $me = shift;

    if( $me->{fd} ){
	$me->{opentime} = $^T;
	$byfile[ fileno( $me->{fd} ) ] = $me;
	$byfile_dbg[ fileno( $me->{fd} ) ] = $me;
	$me->setnbio();
    }else{
	::warning( "basio_init called on invalid fd " .
		   ($me->can('unique') ? $me->unique() : 'unknown') );
    }
}

# { time func text }
sub add_timed_func {
    my $me = shift;
    my @param = @_;
    my( $t );

    $t = { @param };
    $t->{obj} = $me;
    return unless $t->{time} && $t->{time} > $^T;

    # @bytime = sort { $a->{time} <=> $b->{time} } (@bytime, $t);

    # QQQ - profiler says that add_timed_func "needs to be better"
    # sort is O(n log(n)) (but in C), but the data is already sorted
    # is just finding proper spot and inserting faster - O(n) (but perl)
    # Yes - 0.0014s -> 0.0004s

    my $i = 0;
    my $tt = $t->{time};
    foreach my $x (@bytime){
	last if $x && $x->{time} > $tt;
	$i++;
    }

    if( $i && !defined($bytime[$i-1]) ){
	$bytime[$i-1] = $t;
    }else{
	splice @bytime, $i, 0, $t;
    }
}

sub clear_timed_funcs {
    my $me = shift;

    @bytime = grep { $_ && $_->{obj} != $me } @bytime;
}

# all done with fd
sub shutdown {
    my $me = shift;
    my $fd = $me->{fd};
    
    $me->settimeout(0);
    return unless $fd;
    $me->wantread(0);
    $me->wantwrit(0);
    $byfile[ fileno($fd) ] = undef;
    $me->{fd} = undef;
    close $fd;
}

sub recycle {
    my $me = shift;

    $me->shutdown();
    $me->clear_timed_funcs();
}

################################################################

# readable/writable/timeout should be overridden in subclasses if used
sub readable {
    my $me = shift;
    my $class = ref($me);
    
    ::warning("BUG ALERT - read from $class (output-only)! ignoring");
    -1;
}

sub writable {
    my $me = shift;
    my $class = ref($me);

    ::warning("BUG ALERT - write to $class (input-only)! ignoring");
    -1;
}

sub timeout {
    my $me = shift;
    my $class = ref($me);

    ::warning("BUG ALERT - timeout in $class (input-only)! ignoring");
    -1;
}

################################################################
# private object methods
################################################################

sub setnbio {
    my $me = shift;

    my $fh = $me->{fd};
    # QQQ - is this portable?
    fcntl($fh, F_SETFL, O_NDELAY);
}

################################################################
# class methods
################################################################

# return an anonymous filehandle
sub anon_fh {
    do { local *FILEHANDLE };
}

sub closeall {

    foreach my $x (@byfile){
	close $x->{fd} if $x->{fd};
    }
}

################################################################
# private
################################################################

# return ($rfds, $wfds, $to)
# with the re-write of wantread/wantwrit this needs only find the TO
sub selectwhat {
    my( $t, $bt );

    $t = $timeouts[0]->{timeout} if( @timeouts );
    if( @bytime ){
	$bt = $bytime[0]->{time};
	$t  = $bt if( !$t || ($bt < $t) );
    }
    ($rfds, $wfds, $t);
}

# now private
# dispatch readable/writable/timeout based on values returned by select
sub dispatch {
    my $rfd = shift;
    my $wfd = shift;
    my( $i, $n, $nc, $m, $x, $xr, $xw );

    $xr = $rfds; $xw = $wfds;
    # most fd's will not be ready, check 8 at a time
    $nc = int( ($#byfile + 7) / 8 );
    for( $n=0; $n < $nc; $n++ ){

	if( vec($rfd, $n, 8) ){
	    for($i=0; $i<8; $i++){
		$m = $n * 8 + $i;
		if( vec($rfd, $m, 1) ){
		    $x = $byfile[ $m ];
		    if( $x ){
			$x->readable();
		    }else{
			::warning( "BUG? unknown file handle $m returned from select/r - IGNORING" );
			::warning( "...fd $m was set in rfds" )
			    if vec($xr, $m, 1);
			::warning( "...perhaps by $byfile_dbg[$m]=" .
				   ($byfile_dbg[$m]->can('unique') ? 
				    $byfile_dbg[$m]->unique() : 'unknown'))
			    if $byfile_dbg[$m];
		    }
		}
	    }
	}
	
	if( vec($wfd, $n, 8) ){
	    for($i=0; $i<8; $i++){
		$m = $n * 8 + $i;
		if( vec($wfd, $m, 1) ){
		    $x = $byfile[ $m ];
		    if( $x ){
			$x->writable();
		    }else{
			::warning( "BUG? unknown file handle $m returned from select/w - IGNORING" );
			::warning( "...fd $m was set in wfds" )
			    if vec($xw, $m, 1);
			::warning( "...perhaps by $byfile_dbg[$m]=" .
				   ($byfile_dbg[$m]->can('unique') ? 
				    $byfile_dbg[$m]->unique() : 'unknown'))
			    if $byfile_dbg[$m];

		    }
		}
	    }
	}
    }

    while( ($#timeouts != -1) && ($timeouts[0]->{timeout} <= $^T) ){
	$x = shift @timeouts;
	# we can bypass settimeout(0)
	$x->{timeout} = 0;
	$x->timeout();
    }
    
    while( ($#bytime != -1) && (!$bytime[0] || $bytime[0]->{time} <= $^T) ){
	$x = shift @bytime;
	next unless $x;
	$x->{func}->( $x->{obj}, $x->{args} );
    }
}

# aka 'the program'
sub mainloop {
    my %param = @_;
    my( $i, $r, $w, $t, $ti );

    $::idletime = 0;
    $^T = $::TIME = time();
    while(1){
	# chk_schedule();
	($r, $w, $t) = selectwhat();
	if( $t ){
	    $t -= $^T;
	    $t = $param{maxperiod} if $param{maxperiod} && ($t > $param{maxperiod});
	}else{
	    $t = $param{maxperiod};
	}
	$t = 1 if $t < 1;
	
	$^T = $ti = $::TIME = time();
	$i = select($r, $w, undef, $t);
	$^T = $::TIME = time();
	$::idletime += $::TIME - $ti;
	if( $i == -1 ){
	    ::sysproblem( "select failed: $!" ) unless $! == EINTR;
	    next;
	}

	dispatch($r, $w);
	# $param{run}->() if $param{run}; # need to keep the profiler happy
        Prog::reap();
    }
}

# try debugging a scheduling issue
sub chk_schedule {
    my( $lt, $er );

    foreach my $x (@bytime){
	next unless $x;
	$er = 1 if $lt && $lt > $x->{time};
	$lt = $x->{time};
    }
    if( $er ){
	print STDERR "schedule corrupt!\n";
	dump_schedule();
    }
}

sub dump_schedule {
    
    foreach my $x (@bytime){
	if( !$x ){
	    print STDERR "[] empty slot\n";
	    next;
	}
	print STDERR "[", ::xxx_localtime($x->{time}) , "] $x->{text}";
	if( $x->{text} eq "cron" ){
	    print STDERR " - /". ($x->{obj}{cron}{freq} ? $x->{obj}{cron}{freq} : "at") .
		" - $x->{obj}{cron}{text}";
	}
	elsif( $x->{obj}->can('unique') ){
	    print STDERR " - ", $x->{obj}->unique(), $x->{obj};
	}
	print STDERR "\n";
    }
}

sub janitor {
    my( @sched );

    foreach my $x (@bytime ){
	push @sched, $x if $x;
    }
    @bytime = sort {$a->{time} <=> $b->{time}} @sched;
}

################################################################
# these 2 functions provide useful debugging info

sub cmd_files {
    my $ctl = shift;
    my( $i, $x );
    
    $ctl->ok();

    for($i=0; $i<=$#byfile; $i++){
	if( $byfile[$i] && $byfile[$i]{fd} ){
	    $x = $byfile[$i];
	    $ctl->write( "$i $x->{type}  [". ($^T - $x->{opentime}) .
			 "]  $x->{wantread}/$x->{wantwrit}/" .
			 ($x->{timeout} ? $x->{timeout} - $^T: 0) );
	    $ctl->write( " " . $x->unique() ) if $x->can( 'unique' );
	    $ctl->write( "\n" );
	}
    }
    $ctl->write("\n");
}

# What's here? the portrait of a blinking idiot,
# Presenting me a schedule! I will read it.
#   -- Shakespeare, Merchant of Venice
sub cmd_schedule {
    my $ctl = shift;
    my $param = shift;
    
    my( $x, $q, $v );
    
    $ctl->ok();
    $q = $param->{queue};
    $v = $param->{verbose};
    
    foreach $x (@bytime){
	if( !$x ){
	    $ctl->write( "[] empty slot\n" ) if $v;
	    next;
	}
	if( !$q || ($x->{text} eq "cron" && $x->{obj}{cron}{queue} eq $q) ){
	    $ctl->write( "[". ::xxx_localtime($x->{time}) .
			 "] $x->{text}" );
	    
	    if( $x->{text} eq "cron" ){
		$ctl->write( " - /". ($x->{obj}{cron}{freq} ? $x->{obj}{cron}{freq} : "at") .
			     " - $x->{obj}{cron}{text}");
	    }
	    elsif( $v && $x->{obj}->can('unique') ){
		$ctl->write( " - " . $x->{obj}->unique() );
	    }
	    $ctl->write("\n");
	}
    }
    $ctl->write("\n");
}

################################################################
# initialization
################################################################

Control::command_install( 'files', \&cmd_files,    "list all open file descriptors" );
Control::command_install( 'sched', \&cmd_schedule, "list all scheduled tasks" );
Doc::register( $doc );

1;
