# -*- perl -*-

# Copyright (c) 2002 by Jeff Weisberg
# Author: Jeff Weisberg <argus @ tcp4me.com>
# Date: 2002-Apr-02 12:30 (EST)
# Function: crontab class
#
# $Id: Cron.pm,v 1.9 2002/12/25 21:42:22 jaw Exp $

package Cron;
@ISA = qw(BaseIO);

# Time travels in divers paces with divers persons.
# I 'll tell you who Time ambles withal, who Time trots withal,
# who Time gallops withal, and who he stands still withal.
#        -- Shakespeare, As You Like It

$doc = {
    package => __PACKAGE__,
    file    => __FILE__,
    isa     => [@ISA],
    methods => {
    },
    fields  => {
	cron::time => {
	    descr => 'time at which the job should run',
	    attrs => ['internal'],
	},
	cron::freq => {
	    descr => 'how often should the cronjob run',
	    attrs => ['internal'],
	},
	cron::func => {
	    descr => 'ref to func to run',
	    attrs => ['internal'],
	},
	cron::args => {
	    descr => 'arg to pass to func',
	    attrs => ['internal'],
	},
	cron::text => {
	    descr => 'description of job, used for debugging',
	    attrs => ['internal'],
	},
	cron::queue => {
	    descr => 'which queue should job be placed on',
	    attrs => ['internal'],
	    vals  => ['general', 'service'],
	},
      },
};


sub new {
    my $class = shift;
    my %param = @_;
    my( $me, $k );

    $me = {};
    bless $me, $class;
    
    foreach $k (qw(time freq func args text queue)){
	$me->{cron}{$k} = $param{$k};
    }

    if( !$me->{cron}{time} && !$me->{cron}{freq} ){
	::warning( "Invalid crontab spec - IGNORING - $param{text}" );
	return;
    }

    $me->{cron}{queue} ||= 'general';
    
    if( $me->{cron}{freq} ){
	$me->{type} = 'Cronjob';
    }else{
	$me->{type} = 'Atjob';
    }
    
    $me->{cron}{time} ||= 
	# randomly perturb start time if not specified
	# prevents everything running all at once at eg. *:00
	$^T + $me->{cron}{freq} + int(rand($me->{cron}{freq}/2));


    $me->add_timed_func( time => $me->{cron}{time},
			 func => \&cronjob,
			 text => "cron" );
    $me;
}

################################################################
# object methods
################################################################
# Take thy fair hour, Laertes; time be thine,
# And thy best graces spend it at thy will!
#     -- Shakepeare, Hamlet

# job dispatch function
sub cronjob {
    my $me = shift;

    $me->{cron}{func}->( $me->{cron}{args} );
    
    if( $me->{cron}{freq} ){
	# reschedule cronjob

	$me->{cron}{time} += $me->{cron}{freq};
	if( $me->{cron}{time} <= $^T ){
	    $me->{cron}{time} =
		(int(($^T - $me->{cron}{time}) / $me->{cron}{freq}) + 1)
		* $me->{cron}{freq} + $me->{cron}{time};
	}

	$me->add_timed_func( time => $me->{cron}{time},
			     func => \&cronjob,
			     text => "cron" );
    }
}

################################################################
# class global init
################################################################
Doc::register( $doc );

# this cannot be done in BaseIO.pm...
Cron->new( freq => 24*3600,
	   text => 'BaseIO cleanup',
	   func => \&BaseIO::janitor,
	   );

1;
