# -*- perl -*-

# Copyright (c) 2002 by Jeff Weisberg
# Author: Jeff Weisberg <argus @ tcp4me.com>
# Date: 2002-Apr-12 20:06 (EDT)
# Function: Tell someone what happened
#
# $Id: Notify.pm,v 1.33 2003/03/30 15:12:31 jaw Exp $

package Notify;
use NotMe;
use Encode;
use POSIX ('_exit');

my %byid     = (); 
my %lastsent = ();	# by dest
my %queue    = ();	# by dest => {not, tag}
my %unacked  = ();	# by id

# only send so often, otherwise queue
# can be overridden in user-defined notification methods via 'qtime: 1234'
my $QUEUETIME = 120;

# we keep re-sending, but if it comes up, give up after this long
my $UNACK_TO  = 1800;

# if older than this, and no longer active, throw them away
my $OLD_AGE = 3600 * 24 * 5;

# delay escalation, if no longer down for this long
my $DELAY_ESCALATE = 600;

# fields which get saved in save file (and then re-loaded)
my @SAVE_FIELDS =qw(created msg sentcnt state objstate priority autoack
		    ackonup period ackedby ackedat escalated lastsent);

sub new {
    my $obj = shift;
    my( $me, $msg, $aa, $dst, @t );

    return unless $obj->{notify}{sendnotify};

    # skip if in override
    return if $me->{obj}->{ovstatus} eq 'override';

    if( $obj->{status} eq 'down' ){
	$msg = $obj->{notify}{messagedn};
	$aa  = $obj->{notify}{autoack};
    }else{
	$msg = $obj->{notify}{messageup};
	$aa  = 1;	# up messages never need to be acked
    }

    # do not notify if user specified an empty msg
    return unless $msg;
    $msg = expand($msg, $obj);
    
    $me = {
	obj      => $obj,
	msg      => $msg,
	created  => $^T,
	idno     => newid(),
	sentcnt  => 0,
	escalated=> 0,
	state    => 'active',		# active, supressed, acked
	objstate => $obj->{status},
	priority => $obj->{notify}{priority},
	autoack  => $aa,
	period   => $obj->{notify}{renotify},
	ackonup  => $obj->{notify}{ackonup},
	status   => {},
	sendto   => [],
	log      => [],

    };
    bless $me;
    return unless $me->init($obj);
    
    # RSN - supress everything else regarding this object
     
    $me->loggit( 'system', 'created' );
    $me->notify();
}

sub init {
    my $me  = shift;
    my $obj = shift;
    my( $notify );

    # use notifyup, notifydn, or notify?
    $notify = $obj->{notify}{$me->{objstate} eq 'down' ? 'notifydn' : 'notifyup'};
    $notify = $obj->{notify}{notify} unless defined $notify;
    
    # set status of each rcpt
    $me->{sendto} = [ { n => 0, start => 0, who => [] } ];
    foreach my $dst ( split /\s+/, $notify ){
	next if $dst eq 'none';
	$me->{status}{$dst} ||= 'created';
	push @{$me->{sendto}[0]{who}}, $dst;
    }

    # He that sendeth a message by the hand of a fool cutteth off the feet, and drinketh damage.
    #   -- proverbs 26:6
    # no recipients? abort
    unless( @{$me->{sendto}[0]{who}} ){
	# QQQ this is too noisy, maybe re-enable later
	# $me->loggit( 'system', 'no recipient' );
    	return undef;
    }

    # pre-build escalation table
    # A wicked messenger falleth into mischief: but a faithful ambassador is health.
    #   -- proverbs 13:17
    my $n = 1;
    if( $obj->{notify}{escalate} ){
	foreach my $esc ( split /\;\s+/, $obj->{notify}{escalate} ){
	    # each is of form N dest dest dest ...
	    # previously, N was number of pages to escalate
	    # currently, it is elapsed-time in minutes
	    # QQQ - is there a preference?
	    my @a = split /\s+/, $esc;
	    next unless @a;
	    my $nt = shift @a;
	    $nt *= 60;  # minutes -> seconds
	    next unless $nt;
	    
	    $me->{sendto}[$n] = { n => $n, start => $nt, who => [@a] };
	    $n ++;
	}
    }

    push @{$obj->{notify}{list}}, $me;
    $byid{$me->{idno}} = $me;
    
    if( $me->{state} eq 'active' ){
	$unacked{$me->{idno}} = $me;
    }

    # for easy access
    $me->{acl_ntfyack}    = $obj->{acl_ntfyack};
    $me->{acl_ntfydetail} = $obj->{acl_ntfydetail};
    $me->{timezone}       = $obj->{notify}{timezone};
    $me->{mailfrom}       = $obj->{notify}{mail_from};
    
    1;
}

# flush out queued pending messages
END {
    flushqueue();
    foreach my $p (values %byid){
	$p->save();
    }
}

# return a unique id number
sub newid {
    my( $id, $n );

    unless(open(F, "+< $::datadir/notno")){
	::loggit( "cannot open $::datadir/notno: $!", 1 );
    }
    chop( $id = <F> );
    $id ++;
    # make sure not in use
    while( exists $byid{$id} ){
	$id = int(rand( 65535 << (++$n/1000) ));
    }
    
    seek F, 0, 0;
    print F "$id\n";
    close F;

    $id;
}

sub save {
    my $me = shift;
    my $f = "$::datadir/notify/$me->{idno}";

    open( N, "> $f" ) || return $me->loggit( 'system', "cannot open save file: $!", 1 );

    print N "idno: $me->{idno}\n";
    print N "object: ", encode( $me->{obj}->unique() ), "\n";
    foreach my $k (@SAVE_FIELDS){
	my $v = $me->{$k};
	next unless defined $v;

	$v = encode( $v );
	print N "$k: $v\n";
    }

    foreach my $dst (keys %{$me->{status}}){
	print N "status: ", encode($dst), " ", encode($me->{status}{$dst}), "\n";
    }
    
    foreach my $s (@{$me->{sendto}}){
	print N "sendto: $s->{start}";
	foreach my $dst (@{$s->{who}}){
	    print N " ", encode($dst);
	}
	print N "\n";
    }

    foreach my $l (@{$me->{log}}){
	print N "log: $l->{time} ", (encode($l->{who}) || '_') , " ", encode($l->{msg}), "\n";
    }
    
    close N;
}

sub load {
    my $obj = shift;
    my $id  = shift;
    my $f = "$::datadir/notify/$id";
    my $me = { obj => $obj };

    open( N, $f ) || return ::loggit( "cannot open save file '$f': $!", 1 );
    while( <N> ){
	chop;

	next if( /^sendto/ );
	if( /^status:\s+([^\s]+) ([^\s]+)/ ){
	    my($dst, $st) = ($1, $2);
	    $me->{status}{ decode($dst) } = decode($st);
	}
	elsif( /^log:\s+([^\s]+) ([^\s]+) ([^\s]+)/ ){
	    my($t, $w, $m) = ($1, $2, $3);
	    push @{$me->{log}}, { time => $t,
				  who  => (($w eq '_') ? '' : decode($w)),
				  msg  => decode($m)
				  };
	}
	else{
	    my( $k, $v ) = split /:\s*/, $_, 2;
	    $me->{$k} = decode($v);
	}
    }
    close N;
    
    # too old and not still active? toss
    if( $me->{state} ne 'active' && $^T - $me->{created} > $OLD_AGE ){
	unlink $f;
	return;
    }
    
    bless $me;
    $me->init($obj);
}

sub notify {
    my $me  = shift;
    my( $l );
    
    foreach my $dst ( @{$me->{sendto}[0]{who}} ){
	$me->sendorqueue($dst);
    }
    
    $me->{sentcnt} ++;
    $me->{lastsent} = $^T;
}

# I say again! repeated the Pigeon
#   -- Alice in Wonderland
sub renotify {
    my $me = shift;

    # auto-ack if aa and it has already been sent
    return $me->ack() if $me->{autoack};

    for ($l=0; $l<=$me->{escalated}; $l++){
	if( $^T - $me->{created} - $me->{sendto}[$l]{start} < $me->{period} ){
	    # too early to resend the escalation
	    last;
	}
	
        foreach my $dst ( @{$me->{sendto}[$l]{who}} ){
            $me->sendorqueue($dst, 'RESENT');
        }
    }
    
    $me->{sentcnt} ++;
    $me->{lastsent} = $^T;
}

sub escalate {
    my $me = shift;
    my( $n );

    # postpone if no longer down for X minutes
    if( $me->{obj}->{status} ne 'down'
	&& $^T - $me->{obj}->{transtime} > $DELAY_ESCALATE ){
	$me->loggit('system', 'delaying escalation');
	return;
    }
    
    $me->{escalated} ++;
    return if $me->{escalated} >= @{$me->{sendto}};
    
    foreach my $dst ( @{$me->{sendto}[$me->{escalated}]{who}} ){
	$me->sendorqueue($dst );  # 'ESCALATED' will be tagged automatically
    }
}

sub sendorqueue {
    my $me  = shift;
    my $dst = shift;
    my $tag = shift;
    my( $qt );

    $qt = NotMe::qtime($dst);
    $qt = $QUEUETIME unless defined $qt;
    
    if( $lastsent{$dst} && ($lastsent{$dst} + $qt > $^T) ){
	$me->queue($dst, $tag);
    }else{
	$me->transmit($dst, $tag);
    }
}

sub queue {
    my $me  = shift;
    my $dst = shift;
    my $tag = shift;
    
    push @{$queue{$dst}}, {not => $me, tag => $tag};
    
    $me->{status}{$dst} = 'queued';
    $me->loggit( $dst, 'queued' );
    $me->save();
}

# And Hezekiah received the letter from the hand of the messengers, and read it
#   -- isaiah 37:14
sub ack {
    my $me = shift;
    my $who = shift;
    my( $dst, $aap );
    
    return undef if $me->{state} ne 'active';
    unless($who){
	$who = 'auto-ack';
	$aap = 1;
	# we do it this way, and not by looking at me->autoack
	# as in some cases autoack = 1, but the ack is from an
	# override or timeout in which case we want to de-queue
    }
    
    $me->{state}   = 'acked';
    $me->{ackedat} = $^T;
    $me->{ackedby} = $who;
    $me->loggit( $who, 'acked' );

    # remove from queue unless it is an auto-ack
    unless( $aap ){
	foreach $dst (keys %queue){
	    @{$queue{$dst}} = grep { $_->{not}{idno} != $me->{idno} } @{$queue{$dst}};
	    delete $queue{$dst} unless @{$queue{$dst}};
	}
	foreach $dst (keys %{$me->{status}}){
	    $me->{status}{$dst} = "acked by $who";
	}
    }
    
    # remove from unacked
    delete $unacked{ $me->{idno} };
    
    # stats, ...

    
    $me->save();
    
    1;
}



sub supress {
    my $me = shift;
    
    # RSN - supress...
}

sub loggit {
    my $me = shift;
    my $who = shift;
    my $msg = shift;
    my $loudly = shift;
    
    push @{$me->{log}}, { time => $^T, who => $who, msg => $msg };
    $msg  = "<$me->{priority}> $msg" if $me->{priority};
    $msg .= " - $who" if $who;
    $me->{obj}->loggit( msg => $msg,
			tag => "NOTIFY-$me->{idno}",
			lpf => $loudly );
}

sub flushqueue {

    foreach my $dst (keys %queue){
	my @p = map { $_->{not} } @{$queue{$dst}};
	my $p = shift @p;
	next unless $p;
	$p->transmit($dst, undef, @p);
	delete $queue{$dst};
    }
}

# This side is Hiems, Winter, this Ver, the Spring;
# the one maintained by the owl, the other by the
# cuckoo. Ver, begin.
#   -- Shakespeare, Loves Labours Lost

# run by cron every minute
sub maintenance {

    # what should I do with outstanding notifs?
    foreach my $p (values %unacked){
	
	# auto-ack if in override
	if( $p->{obj}->{ovstatus} eq 'override' ){
	    $p->ack('override');
	    next;
	}
	# auto-ack if no longer down and TO
	if( $p->{obj}->{status} ne 'down' && $^T - $p->{obj}->{transtime} > $UNACK_TO ){
	    $p->ack('timeout');
	    next;
	}
	# auto-ack if not down and ackonup is set
	if( $p->{obj}->{status} ne 'down' && $p->{ackonup} ){
	    $p->ack('up');
	}
	
	if( $p->{period} && ($p->{lastsent} + $p->{period} <= $^T) ){
	    # time to resend
	    $p->renotify();
	}

	if( defined($p->{sendto}[ $p->{escalated}+1 ]) &&
	    $^T - $p->{created} >= $p->{sendto}[ $p->{escalated}+1 ]{start} ){
	    # time to escalate
	    $p->escalate();
	}
    }

    # run queue
    foreach my $dst (keys %queue){
	# print STDERR "sendq? $dst\n";
	# time to send?
	my $qt = NotMe::qtime($dst);
	$qt = $QUEUETIME unless defined $qt;
	
	if( $lastsent{$dst} + $qt <= $^T ){
	    # print STDERR "sending q $dst\n";
	    my $p = shift @{$queue{$dst}};
	    my @p = map { $_->{not} } @{$queue{$dst}};
	    # send all queued
	    $p->{not}->transmit($dst, $p->{tag}, @p) if $p;
	    delete $queue{$dst};
	}
    }

}

# And may she speed your footsteps in all good,
# Again began the courteous janitor;
# Come forward then unto these stairs of ours.
#   -- Dante, Divine Comedy

# perform occasional cleanup, etc
sub janitor {

    my( %n_byid, %n_unacked );
    
    foreach my $n (values %byid){
	# if no longer active, and old, => toss
	if( $n->{state} ne 'active' && $^T - $n->{created} > $OLD_AGE ){
	    my $f = "$::datadir/notify/$n->{idno}";
	    unlink $f;
	    $n->{obj}{notify}{list} = [ grep { $_ != $n }
					@{$obj->{notify}{list}} ];
	    $n->{log} = $n->{sendto} = $n->{status} = $n->{obj} = undef
	}else{
	    $n_byid{ $n->{idno} } = $n;
	}
    }
    %byid = (); %byid = %n_byid;

    foreach my $n (values %unacked){
	$n_unacked{ $n->{idno} } = $n;
    }
    %unacked = (); %unacked = %n_unacked;

    # delete old orphaned files
    my $dir = "$::datadir/notify";
    opendir(ND, $dir);
    foreach my $f (readdir(ND)){
	next if $f =~ /^\./;
	next if $byid{$f};
	next if $^T - (stat("$dir/$f"))[9] < $OLD_AGE;
	unlink "$dir/$f";
    }
    closedir ND;
    
}

# On this Iris, fleet as the wind, sped forth to deliver her message.
#   -- Homer, The Iliad
sub transmit {
    my $me  = shift;
    my $dst = shift;
    my $tag = shift;
    my @more = @_;
    my( $msg, $extra );

    # if many, summarize
    if( @more ){
	my $d;
	foreach my $p (@more){
	    $d = 1 if $p->{objstate} eq 'down';
	}
	if( $d ){
	    $msg = $Conf::Config{message_lotsdn} || 'Lots of stuff just went DOWN!';
	}else{
	    $msg = $Conf::Config{message_lotsup} || 'Oodles and oodles just came UP!';
	}
    }else{
	$msg = "$me->{idno} $me->{msg}";
    }

    $extra .= " $tag" if $tag;
    $extra .= " ESCALATED" if $me->{escalated};	# RSN - from $who...

    NotMe::transmit($me, $dst, $msg, $extra)
	unless $Conf::Config{_dont_ntfy};

    $lastsent{$dst} = $^T;
    foreach my $p ( $me, @more ){
	$p->loggit( $dst, 'transmit' );
	$p->{status}{$dst} = 'sent';
	if( $p->{autoack} ){
	    $p->ack();
	}else{
	    $p->save();
	}
    }
}

sub expand {
    my $msg = shift;
    my $obj = shift;

    $msg =~ s/%r/$obj->{srvc}{reason}/g;
    $msg =~ s/%v/$obj->{test}{currvalue}/g;

    $msg;
}

################################################################

sub number_of_notifies { scalar keys %byid }

sub cmd_list_line {
    my $me = shift;
    my $ctl = shift;
    
    $ctl->write("$me->{idno} $me->{objstate} $me->{created} " .
		encode($me->{obj}->filename()) . " " .
		encode($me->{msg}) . "\n");
}

sub cmd_list {
    my $ctl = shift;
    my $param = shift;
    
    $ctl->ok();

    if( $param->{which} eq 'unacked' ){
	# oldest first
	foreach my $p (sort {$a->{created} <=> $b->{created}} values %unacked){
	    $p->cmd_list_line($ctl);
	}
    }
    elsif( $param->{which} eq 'queued' ){
	foreach my $dst (sort keys %queue){
	    # print STDERR "list q $dst\n";
	    foreach my $p (@{$queue{$dst}}){
		$p->{not}->cmd_list_line($ctl);
	    }
	}
    }
    else{
	# newest first
	foreach my $p (sort {$b->{created} <=> $a->{created}} values %byid){
	    $p->cmd_list_line($ctl);
	}
    }
    $ctl->write("\n");
}

sub cmd_detail {
    my $ctl = shift;
    my $param = shift;
    my( $p, $k, $v, $ww, $n );

    $p = $param->{idno};
    if( exists $byid{$p} ){
	$p = $byid{$p};
	$ctl->ok();
	foreach $k (sort keys %$p){
	    $v = $p->{$k};
	    if( ref($v) ){
		$v = "#<REF>";
	    }else{
		$v = encode( $v );
	    }
	    $ctl->write( "$k:\t$v\n" );
	}
	$ctl->write( "object: " . encode( $p->{obj}->unique() ) . "\n" );

	foreach my $w (sort keys %{$p->{status}}){
	    my $we = encode($w);
	    $ww .= " $we";
	    $ctl->write("status $we: " . $p->{status}{$w} . "\n" );
	}
	$ctl->write("statuswho: $ww\n");
	$n = 0;
	foreach my $l (@{$p->{log}}){
	    $ctl->write( "log $n: $l->{time} " . ($l->{who}? encode($l->{who}) : '_') .
			 ' ' . encode($l->{msg}) . "\n" );
	    $n++;
	}
	$ctl->write( "loglines: $n\n");
	$ctl->write( "\n" );
    }else{
	$ctl->bummer(404, 'Notification Not Found');
    }
}

sub cmd_ack {
    my $ctl = shift;
    my $param = shift;
    my( $p, $u );
    
    $p = $param->{idno};
    $u = $param->{user} || 'anonymous';
    
    if( $p eq 'all' ){
	$ctl->ok_n();
	foreach $p (values %unacked){
	    $p->ack( $u );
	}
    }
    elsif( exists $byid{$p} ){
	$ctl->ok_n();
	$byid{$p}->ack( $u );
    }else{
	$ctl->bummer(404, 'Notification Not Found');
    }
}

################################################################

Control::command_install( 'notify_list',     \&cmd_list,
			  'list notifications', 'which' );
Control::command_install( 'notify_detail', \&cmd_detail,
			  'tell about notification', 'idno' );
Control::command_install( 'notify_ack', \&cmd_ack,
			  'ack a notification', 'idno user' );

Cron->new( freq => 60,
	   text => 'Notification maintenance',
	   func => \&maintenance );

Cron->new( freq => 6*3600,
	   text => 'Notification cleanup',
	   func => \&janitor );

1;
