# -*- perl -*-

# Copyright (c) 2002 by Jeff Weisberg
# Author: Jeff Weisberg <argus @ tcp4me.com>
# Date: 2002-Apr-03 18:25 (EST)
# Function: testing of UDP services
#
# $Id: UDP.pm,v 1.39 2003/04/14 16:33:55 jaw Exp $


package UDP;
@ISA = qw(Service);
use Encode;
use Fcntl;
use Socket;
use SNMP;
use DNS;

use POSIX qw(:errno_h);
BEGIN {
    eval { require Socket6; import Socket6; };
}


$doc = {
    package => __PACKAGE__,
    file    => __FILE__,
    isa     => [qw(Service MonEl BaseIO)],
    methods => {},
    html   => 'services',
    fields => {
      udp::hostname => {
	  descr => 'hostname or IP address to test',
	  attrs => ['config', 'inherit'],
      },
      udp::port => {
	  descr => 'UDP port to test',
	  attrs => ['config'],
      },
      udp::addr => {
	  descr => 'encoded IP address',
      },
      udp::srcaddr => {
	  descr => 'local IP address',
	  attrs => ['config', 'inherit'],
	  versn => '3.2',
      },
      udp::srcip => {
	  descr => 'encoded local IP address',
      },
	
      udp::send => {
	  descr => 'text to send once connected',
	  attrs => ['config'],
      },
      udp::rbuffer => {
	  descr => 'read buffer',
      },
      udp::wbuffer => {
	  descr => 'write buffer',
      },
	
    },
};

%config =
(
 # not supported by all radius servers
 RADIUS => {
     port => 1645, send => pack( "CCSx20", 12, $$, 20 ),
     timeout => 10,
 },
 
 SNMP => {
     port => 161,  timeout => 5, 
 },

 NTP => { # RFC 2030
     port => 123,  timeout => 10,
     send => pack("CCCC x44", 0x23,0,0,0),
 },
 'NTP/Stratum' => {
     port => 123,  timeout => 10,
     send => pack("CCCC x44", 0x23,0,0,0),
     unpack => 'xC',
 },
 'NTP/Dispersion' => {
     # root dispersion in seconds
     port => 123,  timeout => 10,
     send => pack("CCCC x44", 0x23,0,0,0),
     unpack => 'x8N', scale => 65536,
 },
 
 # NFS = RFC 1094; RPC = RFC 1057; XDR = RFC 1014
 # NFSv2 - NFSPROC_NULL
 NFS => {
     port => 2049, timeout => 10,
     send => pack( "NN NN NN x16", $$, 0,  2, 100003, 2, 0),
     # xid, type, rpcver, prog, ver, func, cred(flavor, len, null), verf(flavor, len, null)
 },

 NFSv3 => {
     port => 2049, timeout => 10,
     send => pack( "NN NN NN x16", $$, 0,  2, 100003, 3, 0),
     #                                                ^
     #                                  only difference
 },

 );

my $PROTO_UDP = getprotobyname('udp');

sub probe {
    my $name = shift;

    return ( $name =~ /^UDP/ ) ? [ 3, \&config ] : undef;
}

sub config {
    my $me = shift;
    my $cf = shift;
    my( $name, $base );

    bless $me if( ref($me) eq 'Service' );

    $name = $me->{name};
    $name =~ s/^UDP\/?//;
    $base = $name;
    $base =~ s/\/.*// unless $config{$base};

    $me->{srvc}{timeout}     ||= $config{$base}{timeout};
    $me->{udp}{port}         ||= $config{$base}{port};
    $me->{udp}{send}         ||= $config{$base}{send};
    $me->{test}{expect}      ||= $config{$base}{expect};
    $me->{test}{unpack}      ||= $config{$base}{unpack};
    $me->{test}{scale}       ||= $config{$base}{scale};

    $me->{label_right_maybe} ||= $name;

    $me->init_from_config( $doc, 'udp' );

    if( $name ){
	$me->{uname} = "${name}_$me->{udp}{hostname}";
    }else{
	$me->{uname} = "UDP_$me->{udp}{port}_$me->{udp}{hostname}";
    }

    $me->{udp}{addr} = ::resolve( $me->{udp}{hostname} );
    unless( $me->{udp}{addr} ){
	return $cf->error( "cannot resolve host $me->{udp}{hostname}" );
    }
    unless( $me->{udp}{port} ){
	return $cf->error( "Incomplete specification or unknown protocol for Service $name" );
    }
    
    if( $me->{udp}{srcaddr} ){
	$me->{udp}{srcip} = ::resolve($me->{udp}{srcaddr});
	$cf->warning( "Cannot resolve $me->{udp}{srcaddr}" ) unless $me->{udp}{srcip};
    }

    $me;
}

sub start {
    my $me = shift;
    my( $fh, $i, $ip, $ipv );

    $me->{fd} = $fh = BaseIO::anon_fh();
    $ip = $me->{udp}{addr};
    $me->Service::start();
    
    if( length($ip) == 4 ){
	$i = socket($fh, PF_INET,  SOCK_DGRAM, $PROTO_UDP);
    }else{
	$i = socket($fh, PF_INET6, SOCK_DGRAM, $PROTO_UDP);
	$ipv = ' IPv6';
    }
    unless($i){
	my $m = "socket failed: $!";
	::sysproblem( "UDP $m" );
	$me->debug( $m );
	return $me->done();
    }

    $me->baseio_init();
    
    $me->debug( "UDP$ipv start" );
    if( length($ip) == 4 ){
	$i = bind($fh, sockaddr_in(0, $me->{udp}{srcip} || INADDR_ANY) );
    }else{
	# QQQ - Socket6.pm says this is 'inaddr6_any', which is the typo?
	$i = bind($fh, pack_sockaddr_in6(0, $me->{udp}{srcip} || in6addr_any) );
    }
    unless($i){
	my $m = "bind failed: $!";
	::sysproblem( "UDP $m" );
	$me->debug( $m );
	return $me->done();
    }

    # Like sending owls to Athens, as the proverb goes.
    #   -- Plato, Diogenes Laertius
    if( length($ip) == 4 ){
	$i = send( $fh, $me->{udp}{send}, 0, sockaddr_in($me->{udp}{port}, $ip) );
    }else{
	$i = send( $fh, $me->{udp}{send}, 0, pack_sockaddr_in6($me->{udp}{port}, $ip) );
    }
    unless($i){
	# QQQ, should I try to sort out errors, down on some, sysproblem on others?
	$me->isdown( "UDP send failed: $!" );
	return;
    }

    $me->debug( 'UDP sent data' );
    $me->wantread(1);
    $me->wantwrit(0);
    $me->settimeout( $me->{srvc}{timeout} );
    $me->{srvc}{state} = 'waiting';

}

sub readable {
    my $me = shift;
    my( $fh, $i, $l );

    $fh = $me->{fd};
    $i = recv($fh, $l, 8192, 0);
    return $me->isdown( "UDP recv failed: $!" )
	unless defined($i);

    $me->debug( 'UDP recv data' );
    $me->{udp}{rbuffer} = $l;		# for debugging
    
    # yet I will read the writing unto the king, and make known to him the interpretation
    #   -- daniel 5:17

    return $me->generic_test( $l );
}

sub timeout {
    my $me = shift;

    $me->isdown( 'UDP timeout' );
}

################################################################
# and also object methods
################################################################

# sub done {
#     my $me = shift;
# 
#     $me->Service::done();
# }
# 
# sub isdown {
#     my $me = shift;
#     my $reason = shift;
# 
#     $me->Service::isdown($reason);
# }
# 
# sub isup {
#     my $me = shift;
# 
#     $me->Service::isup();
# }

sub about_more {
    my $me = shift;
    my $ctl = shift;

    $me->Service::about_more($ctl);
    $me->more_about_whom($ctl, 'udp');
}

sub webpage_more {
    my $me = shift;
    my $fh = shift;
    
    foreach my $k (qw(hostname port)){
	my $v = $me->{udp}{$k};
	print $fh "<TR><TD>$k</TD><TD>$v</TD></TR>\n" if defined($v);
    }
}

################################################################
# global config
################################################################
Doc::register( $doc );
push @Service::probes, \&probe;

1;

