#!/usr/local/bin/perl
#
# Copyright (c) 2003-2004 Jeremy Kister
# Author: Jeremy Kister <argus-devel @t jeremykister.com>
# Date: 2004-Sep-10 02:05 (EDT)
# Function: send messages via AIM/otc

use strict;
use Net::AIM;
use Getopt::Std;

$| = 1; #flush
my $idle_time=0;
my (%buddies,$sentaway);
chdir('/') || die "could not chdir /: $!\n";

our($opt_u,$opt_g,$opt_d,$opt_s,$opt_p,$opt_D,$opt_b);
getopts('u:g:d:s:p:Db:');

$SIG{USR1} = sub {
	print "received SIGUSR1: running queue\n" if($opt_D);
	$idle_time = 0;
};

# change uid/gid code *STOLEN* from Jeff Weisberg (argusd, http://argus.tcp4me.com)
# Change to nobody if you'd like, or to a user who has argusctl permissions.
if(defined($opt_g)){
	my $gid = (getgrnam($opt_g))[2];
	$gid = $opt_g if( !defined($gid) && $opt_g =~ /^\d+$/ );
	slowdie("invalid group for -g option. aborting.") unless defined $gid;
	$( = $gid; $) = $gid;
}
if(defined($opt_u)){
	my $uid = (getpwnam($opt_u))[2];
	$uid = $opt_u if( !defined($uid) && $opt_u =~ /^\d+$/ );
	slowdie("invalid user for -u option. aborting.") unless defined $uid;

	# fix /tmp/aaimqueue/ if owned by anyone other than $opt_u
	if(-d '/tmp/aaimqueue'){
		my $diruid = (stat('/tmp/aaimqueue/'))[4];
		if($uid != $diruid){
			warn "/tmp/aaimqueue not owned by $opt_u: fixing..\n";
			chown($uid,$(,'/tmp/aaimqueue/') || slowdie("cannot chown /tmp/aaimqueue/: $!");
		}
	}else{
		warn "/tmp/aaimqueue/ does not exist - creating..\n";
		mkdir('/tmp/aaimqueue',0700) || slowdie("cannot create /tmp/aaimqueue/: $!");
		chown($uid,$(,'/tmp/aaimqueue/') || slowdie("cannot chown /tmp/aaimqueue/: $!");
	}
	# fix /tmp/aaimqueue/ files if owned by anyone other than $opt_u, (so we dont keep dying)
	if(opendir(D, '/tmp/aaimqueue/')){
		foreach my $file (grep {!/^\./} readdir D){
			my $fileuid = (stat("/tmp/aaimqueue/${file}"))[4];
			if($uid != $fileuid){
				warn "/tmp/aaimqueue/${file} not owned by ${opt_u}: fixing..\n";
				chown($uid,$(,"/tmp/aaimqueue/${file}") || slowdie("cannot chown $file: $!");
			}
		}
		closedir D;
	}

	$! = 0;
	$< = $> = $uid;
	die "unable to change uid: $!\n" if $!;
}

unless(defined($opt_s) && defined($opt_p) && defined($opt_b)){
	print "syntax:\n",
	      "	aim.pl [-u <username>] [-g <group>] [-D] [-d </path/argus/sbin>] -s <screenname> -p <password> -b <Buddies>\n",
	      "	options:\n",
	      "		-u	username to run as\n",
 	      "		-g	group to run as\n",
	      "		-D	print debugging info\n",
	      "		-d	alternative path to argus sbin directory (default: /usr/local/sbin)\n",
	      "		-s	AIM screenname\n",
	      "		-p	AIM password\n",
	      "		-b	list of buddies (separate multiples by comma, no spaces in usernames)\n\n";
	exit;
}

my $sbin = ($opt_d) ? $opt_d : '/usr/local/sbin';
$opt_b =~ s/\ //g;
my @buddies = split(/,/, lc(${opt_b}));

print "Starting Server...\n";

my $aim = new Net::AIM;
$aim->newconn(Screenname => $opt_s,
              Password   => $opt_p)
                or slowdie("Error: Cannot connect to the AIM server!");
$aim->debug(0);
my $conn = $aim->getconn();

print "connected to aim server as ${opt_s}.\n" if($opt_D);

# Set up the handlers for commands issued by the server.
$conn->set_handler('im_in', \&on_im);
$conn->set_handler('error', \&on_error);
$conn->set_handler('eviled', \&on_evil);
$conn->set_handler('config', \&on_config);
$conn->set_handler('update_buddy', \&on_update_buddy);

# add our buddies
$aim->add_buddy(0,'Buddies',@buddies);

for(1..6){
	$aim->do_one_loop(); # first few polls dont seem to do anything..
}

while($conn){ # exit if connection is lost
	# check to see if we have incoming messages waiting
	$aim->do_one_loop();

	# check to see if we have outgoing messages waiting
	if(opendir(D, '/tmp/aaimqueue/')){
		foreach my $file (sort {$a<=>$b} grep {!/^\./} readdir D){
			if(open(F, "/tmp/aaimqueue/${file}")){
				chop(my $line=<F>); # 1st line is recipient(s)
				$line =~ s/\ //g;
				my @recipients = split(/,/, lc($line));
				my $sendmsg;
				while(<F>){
					$sendmsg .= $_;
				}
				close F;
				chomp($sendmsg);

				# stop AIM/trillian client from changing things like :Ping to <happy face>ing
				$sendmsg =~ s/:/ -> /g;
				foreach my $recipient (@recipients){
					if(exists($buddies{$recipient})){
						print "${opt_s} -> ${recipient}: ${sendmsg}\n" if($opt_D);
						my $erecip = $aim->normalize($recipient);
						$aim->send_im($erecip, $sendmsg);
						sleep 1;
					}else{
						print "skipping $recipient (not online)\n" if($opt_D);
					}
				}
				unlink("/tmp/aaimqueue/${file}") || slowdie("cannot unlink /tmp/aaimqueue/${file}: $!");
				$idle_time=0;
			}else{
				slowdie("cannot open /tmp/aaimqueue/${file}");
			}
		}
		closedir D;
	}else{
		warn "could not open /tmp/aaimqueue/: $!\n";
	}

	if($idle_time <= 5){
		if(defined($sentaway)){
			print "WHO!? WHA!? -- damn it.. i hate being woken up :)\n" if($opt_D);
			$aim->set_away();
			$sentaway=undef;
		}
		sleep 1;
		$idle_time++;
	}else{
		sleep 5;
		$idle_time += 5;
		unless(defined($sentaway)){
			if($idle_time > 180){
				print "<yawn> -- taking a nap..\n" if($opt_D);
				$aim->set_away($idle_time);
				$sentaway=1;
			}
		}
	}
}
print "lost connection to AIM!\n";

sub on_config {
	# called when the bot is being loaded.
	my ($self, $evt, $from, $to) = @_;
	
	my $str = shift @{$evt->args()};
	$self->set_config_str($str, 1);

	#Set up my profile.
	$self->set_info("Argus AIM v.95b");

	$self->send_config();
}

sub on_error {
	# called when an error occurs while communicating w/ TOC.
	my ($self, $evt) = @_;
	my ($error, @stuff) = @{$evt->args()};

	#Translate the error number into plain English.
	my $errstr = $evt->trans($error);

	#Filter the error string for the content we need.
	$errstr =~ s/\$(\d+)/$stuff[$1]/ge;

	print "AIM ERROR: $errstr\n";

	if($errstr =~ /A message has been dropped/){
		warn "sleeping 10..\n";
		sleep(10);
	}elsif($errstr =~ /Wait\s+(\d+)\s+/){
		my $seconds=($1*60);
		warn "Sleeping $seconds\n";
		sleep($seconds+5);
	}
}

sub on_evil {
	# when the bot recieves a warning.
	my ($self, $evt, $from, $to) = @_;
	my ($level, $bastard) = @{$evt->args};

	print "My warning level is $level %.";

	#Remove spaces from $bastard; make it lowercase.
	$bastard = lc ($bastard);
	$bastard =~ s/ //g;

	#Warn and block him.
	$aim->evil($bastard, 0);
	$aim->add_deny(1, 'Buddies', $bastard);
}

sub on_update_buddy {
	my ($self, $evt, $from, $to) = @_;
	my ($bud, $online, $evil, $event_time, $idle_amount, $user_class) = @{$evt->args()};

	$bud =~ s/\ //g;
	$bud = lc($bud);
	#lets print out when users leave and when they enter;
	if(($online eq 'T') && (! exists($buddies{$bud}))){
		$buddies{$bud} = $event_time;
		print "$bud has signed on at " . scalar localtime($event_time) . "\n" if($opt_D);
	}elsif($online eq 'F'){
		delete $buddies{$bud};
		print "$bud has signed off at " . scalar localtime($event_time)  . "\n" if($opt_D);
	}
}

sub on_im {
	# called when the bot recieves an IM.
	my ($aim, $evt, $from, $to) = @_;
	my $args = $evt->args();
	my ($sender, $friend, $msg) = @$args;

	$sender =~ s/\ //g;
	$sender = lc($sender);

	#Format the message without HTML.
	$msg =~ s/<(.|\n)+?>//g;

	print "${sender}: ${msg}\n" if($opt_D);

	# we can do fun stuff here (is secure as AIM buddy list)
	if(exists($buddies{$sender})){
		my $send;
		if($msg =~ /^ACK:\s*((\d+|all))$/){
			my $pageid=$1;
			$send="Acking PageID: $pageid\n";
			$send .= `$sbin/argusctl notify_ack idno=$pageid AIM::$buddies{$sender}`;
		}elsif($msg eq 'argusctl status'){
			$send=`$sbin/argusctl status`;
		}elsif($msg eq 'ping'){
			$send='pong';
		}elsif($msg eq 'whoson'){
			while(my($key,$value) = each %buddies){
				$send .= "$key -> $value\n";
			}
		}
		if(defined($send)){
			my $erecip = $aim->normalize($sender);
			$aim->send_im($erecip,$send);
			print "${opt_s} -> ${sender}: ${send}\n" if($opt_D);
			$idle_time=0; #only reset idle if a valid command was sent to us
		}
	}
}

sub slowdie {
	my $err = shift;
	warn "$err\n";
	sleep 5;
	exit 1;
}

